// David Eberly, Geometric Tools, Redmond WA 98052
// Copyright (c) 1998-2025
// Distributed under the Boost Software License, Version 1.0.
// https://www.boost.org/LICENSE_1_0.txt
// https://www.geometrictools.com/License/Boost/LICENSE_1_0.txt
// File Version: 8.0.2025.05.10

#pragma once

// Create the lookup table for the Marching Cubes algorithm that is used to
// extract a triangular mesh that represents a level surface of a 3D image
// sampled on a regular lattice. The assumption is that no data sample is
// zero, which allows us to have a table with 256 entries: 2 signs per
// sample, 8 samples per volume element (voxel). Each entry corresponds to
// the pattern of 8 signs at the corners of a voxel. The signs are stored as
// bits (b7,b6,b5,b4,b3,b2,b1,b0). The bit assignments to voxel corners is
//   b0 = (x,y,z),   b1 = (x+1,y,z),   b2 = (x,y+1,z),   b3 = (x+1,y+1,z)
//   b4 = (x,y,z+1), b5 = (x+1,y,z+1), b6 = (x,y+1,z+1), b7 = (x+1,y+1,z+1)
// If a bit is zero, then the voxel value at the corresponding corner is
// positive; otherwise, the bit is one and the value is negative. The
// triangles are counterclockwise ordered according to an observer viewing
// the triangle from the negative side of the level surface.

#include <algorithm>
#include <array>
#include <cstdint>
#include <string>

namespace gte
{
    template <typename IndexType>
    class MarchingCubes
    {
    public:
        // Construction and destruction.
        virtual ~MarchingCubes() = default;

        MarchingCubes()
        {
            // Create the lookup table.
            for (mEntry = 0; mEntry < 256; ++mEntry)
            {
                (this->*ConfigurationTable()[mEntry].F)(ConfigurationTable()[mEntry].index);
            }
        }

        struct Topology
        {
            // All members are set to zeros.
            Topology()
                :
                numVertices(0),
                numTriangles(0),
                vpair{},
                itriple{}
            {
                std::fill(vpair.begin(), vpair.end(), std::array<IndexType, 2>{ 0, 0 });
                std::fill(itriple.begin(), itriple.end(), std::array<IndexType, 3>{ 0, 0, 0 });
            }

            static size_t constexpr maxVertices = 12;
            static size_t constexpr maxTriangles = 5;
            IndexType numVertices;
            IndexType numTriangles;
            std::array<std::array<IndexType, 2>, maxVertices> vpair;
            std::array<std::array<IndexType, 3>, maxTriangles> itriple;
        };

        // The entry must be in {0..255}.
        inline Topology const& GetTable(size_t entry) const
        {
            return TopologyTable()[entry];
        }

        // The table has 256 entries, each 41 integers, stored as
        // table[256][41]. The return value is a pointer to the table's
        // first IndexType element.
        inline IndexType const* GetTable() const
        {
            return reinterpret_cast<IndexType const*>(TopologyTable().data());
        }

        // The pre-built topology table that is generated by the constructor.
        // This is for reference in case you want to have a GPU-based
        // implementation where you load the table as a GPU resource.
        static std::array<std::array<IndexType, 41>, 256> const& GetPrebuiltTable()
        {
            static std::array<std::array<IndexType, 41>, 256> topologyTable =
            { {
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 0, 2, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 2, 6, 2, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 1, 5, 0, 1, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 3, 7, 1, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 2, 6, 3, 7, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 5, 4, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 1, 3, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 4, 4, 5, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 0, 2, 2, 6, 2, 3, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 4, 6, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 2, 3, 3, 7, 1, 3, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 1, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 1, 2, 3, 3, 7, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 2, 6, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 2, 6, 3, 7, 1, 3, 0, 1, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 4, 0, 2, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 2, 2, 6, 2, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 4, 2, 6, 2, 3, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 3, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 0, 4, 2, 6, 2, 3, 1, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 5, 7, 4, 5, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 4, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 1, 3, 0, 2, 2, 6, 3, 7, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 0, 4, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 2, 1, 3, 5, 7, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 5, 7, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 3, 7, 5, 7, 4, 6, 0, 4, 0, 1, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 3, 7, 1, 3, 0, 2, 2, 6, 1, 5, 5, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 3, 7, 2, 6, 4, 6, 5, 7, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 4, 6, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 2, 0, 1, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 2, 1, 3, 1, 5, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 6, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 2, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 3, 7, 1, 3, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 2, 3, 3, 7, 1, 3, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 3, 7, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 6, 7, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 3, 7, 1, 5, 0, 1, 0, 2, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 6, 7, 2, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 5, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 2, 6, 0, 2, 1, 3, 1, 5, 4, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 1, 4, 5, 6, 7, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 6, 0, 4, 4, 5, 6, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 8, 4, 1, 5, 0, 1, 2, 3, 3, 7, 0, 4, 4, 5, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 6, 7, 4, 5, 1, 5, 3, 7, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 1, 5, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 4, 5, 1, 5, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 5, 0, 1, 1, 3, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 3, 6, 7, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 8, 4, 4, 6, 6, 7, 2, 3, 0, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 5, 7, 1, 3, 2, 3, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 3, 6, 7, 2, 6, 4, 6, 4, 5, 1, 5, 5, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 12, 4, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0, 0 },
                { 8, 4, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 4, 6, 0, 4, 4, 5, 5, 7, 3, 7, 6, 7, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 8, 4, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 5, 4, 6, 0, 2, 0, 1, 4, 5, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 5, 3, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 1, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 4, 1, 5, 5, 7, 6, 7, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 2, 3, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 1, 3, 2, 3, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 5, 2, 3, 0, 1, 0, 4, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 2, 3, 0, 2, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 7, 5, 1, 5, 0, 4, 0, 2, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 1, 5, 0, 1, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 5, 0, 1, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 5, 0, 4, 0, 2, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 2, 3, 0, 2, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 3, 0, 1, 0, 4, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 4, 2, 1, 3, 2, 3, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 3, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 1, 5, 0, 4, 0, 2, 2, 3, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 5, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 6, 0, 2, 0, 1, 4, 5, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 3, 4, 6, 0, 4, 4, 5, 5, 7, 3, 7, 6, 7, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 12, 4, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0, 0 },
                { 9, 5, 6, 7, 2, 6, 4, 6, 4, 5, 1, 5, 5, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 3, 5, 7, 1, 3, 2, 3, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 5, 4, 6, 0, 2, 0, 1, 6, 7, 5, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 8, 4, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 6, 7, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 8, 4, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 4, 5, 0, 1, 1, 3, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 5, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 4, 5, 1, 5, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 5, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 6, 7, 4, 5, 1, 5, 3, 7, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 6, 2, 3, 0, 1, 0, 4, 3, 7, 6, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 8, 4, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 2, 6, 0, 4, 4, 5, 6, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 4, 2, 0, 1, 2, 3, 6, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 5, 1, 3, 0, 2, 2, 6, 6, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 4, 5, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 6, 7, 2, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 6, 7, 4, 6, 0, 2, 0, 1, 1, 5, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 3, 7, 6, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 3, 7, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 5, 2, 3, 3, 7, 1, 3, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 3, 7, 1, 3, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 4, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 2, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 6, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 0, 2, 1, 3, 1, 5, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 1, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 6, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 3, 7, 2, 6, 4, 6, 5, 7, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 0, 2, 1, 3, 1, 5, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 5, 3, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 3, 7, 2, 3, 0, 1, 0, 4, 4, 6, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 5, 7, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 0, 2, 4, 6, 5, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 0, 4, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 1, 3, 0, 2, 2, 6, 3, 7, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 5, 7, 3, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 9, 5, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 5, 7, 4, 5, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 4, 2, 3, 2, 6, 0, 4, 4, 5, 5, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 1, 3, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 0, 4, 2, 6, 2, 3, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 0, 2, 2, 6, 2, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 4, 0, 2, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 1, 3, 3, 7, 2, 6, 4, 6, 4, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 2, 6, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 1, 2, 3, 3, 7, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 0, 1, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 2, 3, 3, 7, 1, 3, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 3, 2, 3, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 9, 5, 0, 2, 2, 6, 2, 3, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 5, 3, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 4, 4, 5, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 5, 3, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 1, 1, 3, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 4, 5, 4, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 1, 5, 3, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 3, 7, 1, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 1, 5, 0, 1, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 2, 6, 2, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 0, 2, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
            } };
            return topologyTable;
        }

        // Get the configuration type for the voxel, which is one of the
        // string names of the 'void Bits* (IndexType[8])' functions.
        static std::string GetConfigurationType(size_t entry)
        {
            if (entry < 256)
            {
                return ConfigurationString()[ConfigurationTable()[entry].type];
            }
            return "";
        }

    protected:
        // Support for lookup construction and access.
        // mTable[i][0] = numVertices
        // mTable[i][1] = numTriangles
        // mTable[i][2..25] = pairs of corner indices (maximum of 12 pairs)
        // mTable[i][26..40] = triples of indices (maximum of 5 triples)
        static std::array<Topology, 256>& TopologyTable()
        {
            static std::array<Topology, 256> topologyTable;
            return topologyTable;
        }

        // The constructor iterates mEntry from 0 to 255 and calls configuration
        // functions, each calling SetTable(...).  The mEntry value is the table
        // index to be used.
        size_t mEntry;

        void SetTable(size_t numV, IndexType const* vpair, size_t numT, IndexType const* itriple)
        {
            // The item is already zeroed in the constructor.
            Topology& topology = TopologyTable()[mEntry];
            topology.numVertices = static_cast<IndexType>(numV);
            topology.numTriangles = static_cast<IndexType>(numT);

            // Store vertex pairs with minimum index occurring first.
            for (size_t i = 0; i < numV; ++i, vpair += 2)
            {
                topology.vpair[i][0] = std::min(vpair[0], vpair[1]);
                topology.vpair[i][1] = std::max(vpair[0], vpair[1]);
            }

            // Store triangle triples as is.
            for (size_t i = 0; i < numT; ++i, itriple += 3)
            {
                topology.itriple[i] = { itriple[0], itriple[1], itriple[2] };
            }
        }

        // The precomputed information about the 256 configurations for voxels.
        void Bits0(std::array<IndexType, 8> const&)
        {
            SetTable(0, nullptr, 0, nullptr);
        }

        void Bits1(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 3;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0]
            };

            size_t constexpr numT = 1;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits7(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 3;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0]
            };

            size_t constexpr numT = 1;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 2, 1
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits2Edge(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 4;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[0],
                index[2], index[0],
                index[3], index[1],
                index[5], index[1]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits6Edge(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 4;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[0],
                index[2], index[0],
                index[3], index[1],
                index[5], index[1]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 2, 1,
                0, 3, 2
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits2FaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[2], index[3],
                index[7], index[3],
                index[1], index[3]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                3, 4, 5
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits6FaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[2], index[3],
                index[7], index[3],
                index[1], index[3]
            };

            // Not the reverse ordering from Bit2FaceDiag due to ambiguous face
            // handling.
            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                1, 0, 5,
                1, 5, 4,
                1, 4, 3,
                1, 3, 2
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits2BoxDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[3], index[7],
                index[6], index[7],
                index[5], index[7]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                3, 4, 5
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits6BoxDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[3], index[7],
                index[6], index[7],
                index[5], index[7]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 2, 1,
                3, 5, 4
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits3SameFace(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 5;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[0],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            size_t constexpr numT = 3;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits5SameFace(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 5;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[0],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            size_t constexpr numT = 3;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 2, 1,
                0, 3, 2,
                0, 4, 3
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits3EdgeFaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 7;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[1],
                index[4], index[5],
                index[4], index[6],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3]
            };

            size_t constexpr numT = 3;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                4, 5, 6
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits5EdgeFaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 7;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[1],
                index[4], index[5],
                index[4], index[6],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3]
            };

            // Not the reverse ordering from Bit3EdgeFaceDiag due to ambiguous face
            // handling.
            size_t constexpr numT = 5;
            std::array<IndexType, 3 * numT> itriple =
            {
                5, 0, 6,
                5, 1, 0,
                5, 2, 1,
                5, 3, 2,
                5, 4, 3
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits3FaceDiagFaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 9;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            size_t constexpr numT = 3;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                3, 4, 5,
                6, 7, 8
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits5FaceDiagFaceDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 9;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            // Not the reverse ordering from Bit3FaceDiagFaceDiag due to ambiguous face
            // handling.
            size_t constexpr numT = 5;
            std::array<IndexType, 3 * numT> itriple =
            {
                1, 3, 2,
                1, 4, 3,
                1, 7, 4,
                1, 8, 7,
                0, 5, 6
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4SameFace(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 4;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[4],
                index[2], index[6],
                index[3], index[7],
                index[1], index[5]
            };

            size_t constexpr numT = 2;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4FaceEdge(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[5],
                index[4], index[6],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4FaceFaceDiagL(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[5],
                index[0], index[4],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[5], index[7]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4FaceFaceDiagR(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 6;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[4], index[6],
                index[6], index[7],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5],
                index[0], index[4]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4FaceBoxDiag(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 8;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[4],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5],
                index[6], index[7],
                index[5], index[7],
                index[3], index[7]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                5, 6, 7
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4EdgeEdgePara(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 8;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[4],
                index[0], index[2],
                index[1], index[3],
                index[1], index[5],
                index[2], index[6],
                index[4], index[6],
                index[5], index[7],
                index[3], index[7]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                0, 2, 3,
                4, 5, 6,
                4, 6, 7
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        void Bits4EdgeEdgePerp(std::array<IndexType, 8> const& index)
        {
            size_t constexpr numV = 12;
            std::array<IndexType, 2 * numV> vpair =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[6],
                index[4], index[6],
                index[6], index[7],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            size_t constexpr numT = 4;
            std::array<IndexType, 3 * numT> itriple =
            {
                0, 1, 2,
                3, 4, 5,
                6, 7, 8,
                9, 10, 11
            };

            SetTable(numV, vpair.data(), numT, itriple.data());
        }

        // Configuration types.
        static size_t constexpr CTBits0 = 0;
        static size_t constexpr CTBits1 = 1;
        static size_t constexpr CTBits7 = 2;
        static size_t constexpr CTBits2Edge = 3;
        static size_t constexpr CTBits6Edge = 4;
        static size_t constexpr CTBits2FaceDiag = 5;
        static size_t constexpr CTBits6FaceDiag = 6;
        static size_t constexpr CTBits2BoxDiag = 7;
        static size_t constexpr CTBits6BoxDiag = 8;
        static size_t constexpr CTBits3SameFace = 9;
        static size_t constexpr CTBits5SameFace = 10;
        static size_t constexpr CTBits3EdgeFaceDiag = 11;
        static size_t constexpr CTBits5EdgeFaceDiag = 12;
        static size_t constexpr CTBits3FaceDiagFaceDiag = 13;
        static size_t constexpr CTBits5FaceDiagFaceDiag = 14;
        static size_t constexpr CTBits4SameFace = 15;
        static size_t constexpr CTBits4FaceEdge = 16;
        static size_t constexpr CTBits4FaceFaceDiagL = 17;
        static size_t constexpr CTBits4FaceFaceDiagR = 18;
        static size_t constexpr CTBits4FaceBoxDiag = 19;
        static size_t constexpr CTBits4EdgeEdgePara = 20;
        static size_t constexpr CTBits4EdgeEdgePerp = 21;
        static size_t constexpr CTNumTypes = 22;

        using Function = void (MarchingCubes<IndexType>::*)(std::array<IndexType, 8> const&);

        struct Configuration
        {
            size_t type;
            Function F;
            std::array<IndexType, 8> const& index;
        };

        static std::array<Configuration, 256>& ConfigurationTable()
        {
            static std::array<Configuration, 256> configuration =
            { {
                    /*00000000*/{ CTBits0, &MarchingCubes::Bits0, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00000001*/{ CTBits1, &MarchingCubes::Bits1, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00000010*/{ CTBits1, &MarchingCubes::Bits1, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00000011*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00000100*/{ CTBits1, &MarchingCubes::Bits1, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00000101*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00000110*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00000111*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00001000*/{ CTBits1, &MarchingCubes::Bits1, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00001001*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00001010*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00001011*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00001100*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00001101*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00001110*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00001111*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00010000*/{ CTBits1, &MarchingCubes::Bits1, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*00010001*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*00010010*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*00010011*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*00010100*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*00010101*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*00010110*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00010111*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00011000*/{ CTBits2BoxDiag, &MarchingCubes::Bits2BoxDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00011001*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00011010*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*00011011*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00011100*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*00011101*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00011110*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00011111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*00100000*/{ CTBits1, &MarchingCubes::Bits1, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*00100001*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*00100010*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*00100011*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*00100100*/{ CTBits2BoxDiag, &MarchingCubes::Bits2BoxDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00100101*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*00100110*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00100111*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00101000*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*00101001*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*00101010*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*00101011*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*00101100*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 3, 1, 7, 5, 2, 0, 6, 4 }},
                    /*00101101*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*00101110*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00101111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*00110000*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*00110001*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*00110010*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*00110011*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*00110100*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*00110101*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*00110110*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*00110111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*00111000*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*00111001*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*00111010*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*00111011*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*00111100*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*00111101*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*00111110*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*00111111*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*01000000*/{ CTBits1, &MarchingCubes::Bits1, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*01000001*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*01000010*/{ CTBits2BoxDiag, &MarchingCubes::Bits2BoxDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*01000011*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*01000100*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*01000101*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*01000110*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*01000111*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*01001000*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*01001001*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*01001010*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*01001011*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*01001100*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*01001101*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*01001110*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*01001111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*01010000*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*01010001*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*01010010*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*01010011*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*01010100*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*01010101*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*01010110*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*01010111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*01011000*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*01011001*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*01011010*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*01011011*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*01011100*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*01011101*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*01011110*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*01011111*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*01100000*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*01100001*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*01100010*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*01100011*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*01100100*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*01100101*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*01100110*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*01100111*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*01101000*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*01101001*/{ CTBits4EdgeEdgePerp, &MarchingCubes::Bits4EdgeEdgePerp, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*01101010*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*01101011*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01101100*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*01101101*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*01101110*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01101111*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*01110000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01110001*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01110010*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*01110011*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*01110100*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01110101*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 3, 1, 7, 5, 2, 0, 6, 4 }},
                    /*01110110*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*01110111*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*01111000*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*01111001*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*01111010*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*01111011*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*01111100*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*01111101*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*01111110*/{ CTBits6BoxDiag, &MarchingCubes::Bits6BoxDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*01111111*/{ CTBits7, &MarchingCubes::Bits7, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*10000000*/{ CTBits1, &MarchingCubes::Bits1, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*10000001*/{ CTBits2BoxDiag, &MarchingCubes::Bits2BoxDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*10000010*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*10000011*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*10000100*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*10000101*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*10000110*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*10000111*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*10001000*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10001001*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*10001010*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 3, 1, 7, 5, 2, 0, 6, 4 }},
                    /*10001011*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*10001100*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10001101*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*10001110*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*10001111*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*10010000*/{ CTBits2FaceDiag, &MarchingCubes::Bits2FaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*10010001*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*10010010*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*10010011*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*10010100*/{ CTBits3FaceDiagFaceDiag, &MarchingCubes::Bits3FaceDiagFaceDiag, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*10010101*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*10010110*/{ CTBits4EdgeEdgePerp, &MarchingCubes::Bits4EdgeEdgePerp, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*10010111*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*10011000*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*10011001*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*10011010*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 3, 1, 7, 5, 2, 0, 6, 4 }},
                    /*10011011*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*10011100*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10011101*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*10011110*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*10011111*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*10100000*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*10100001*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*10100010*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*10100011*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*10100100*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*10100101*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*10100110*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*10100111*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*10101000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*10101001*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*10101010*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*10101011*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*10101100*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10101101*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*10101110*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*10101111*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*10110000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*10110001*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*10110010*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*10110011*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*10110100*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 5, 4, 7, 6, 1, 0, 3, 2 }},
                    /*10110101*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10110110*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*10110111*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 3, 7, 2, 6, 1, 5, 0, 4 }},
                    /*10111000*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*10111001*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*10111010*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*10111011*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*10111100*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*10111101*/{ CTBits6BoxDiag, &MarchingCubes::Bits6BoxDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*10111110*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*10111111*/{ CTBits7, &MarchingCubes::Bits7, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*11000000*/{ CTBits2Edge, &MarchingCubes::Bits2Edge, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*11000001*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*11000010*/{ CTBits3EdgeFaceDiag, &MarchingCubes::Bits3EdgeFaceDiag, { 7, 3, 5, 1, 6, 2, 4, 0 }},
                    /*11000011*/{ CTBits4EdgeEdgePara, &MarchingCubes::Bits4EdgeEdgePara, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11000100*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*11000101*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*11000110*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 6, 2, 7, 3, 4, 0, 5, 1 }},
                    /*11000111*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*11001000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*11001001*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*11001010*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 7, 6, 3, 2, 5, 4, 1, 0 }},
                    /*11001011*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*11001100*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 2, 3, 6, 7, 0, 1, 4, 5 }},
                    /*11001101*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*11001110*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*11001111*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*11010000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*11010001*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*11010010*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*11010011*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 3, 1, 7, 5, 2, 0, 6, 4 }},
                    /*11010100*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 6, 4, 2, 0, 7, 5, 3, 1 }},
                    /*11010101*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 1, 5, 3, 7, 0, 4, 2, 6 }},
                    /*11010110*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11010111*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*11011000*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 6, 7, 4, 5, 2, 3, 0, 1 }},
                    /*11011001*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*11011010*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*11011011*/{ CTBits6BoxDiag, &MarchingCubes::Bits6BoxDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*11011100*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*11011101*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 5, 1, 4, 0, 7, 3, 6, 2 }},
                    /*11011110*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*11011111*/{ CTBits7, &MarchingCubes::Bits7, { 5, 7, 1, 3, 4, 6, 0, 2 }},
                    /*11100000*/{ CTBits3SameFace, &MarchingCubes::Bits3SameFace, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*11100001*/{ CTBits4FaceBoxDiag, &MarchingCubes::Bits4FaceBoxDiag, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*11100010*/{ CTBits4FaceFaceDiagL, &MarchingCubes::Bits4FaceFaceDiagL, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*11100011*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 2, 6, 0, 4, 3, 7, 1, 5 }},
                    /*11100100*/{ CTBits4FaceFaceDiagR, &MarchingCubes::Bits4FaceFaceDiagR, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*11100101*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*11100110*/{ CTBits5EdgeFaceDiag, &MarchingCubes::Bits5EdgeFaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11100111*/{ CTBits6BoxDiag, &MarchingCubes::Bits6BoxDiag, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*11101000*/{ CTBits4FaceEdge, &MarchingCubes::Bits4FaceEdge, { 7, 5, 6, 4, 3, 1, 2, 0 }},
                    /*11101001*/{ CTBits5FaceDiagFaceDiag, &MarchingCubes::Bits5FaceDiagFaceDiag, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*11101010*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 0, 2, 4, 6, 1, 3, 5, 7 }},
                    /*11101011*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*11101100*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 0, 4, 1, 5, 2, 6, 3, 7 }},
                    /*11101101*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 1, 0, 5, 4, 3, 2, 7, 6 }},
                    /*11101110*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 4, 0, 6, 2, 5, 1, 7, 3 }},
                    /*11101111*/{ CTBits7, &MarchingCubes::Bits7, { 4, 5, 0, 1, 6, 7, 2, 3 }},
                    /*11110000*/{ CTBits4SameFace, &MarchingCubes::Bits4SameFace, { 4, 6, 5, 7, 0, 2, 1, 3 }},
                    /*11110001*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*11110010*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*11110011*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*11110100*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*11110101*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*11110110*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11110111*/{ CTBits7, &MarchingCubes::Bits7, { 3, 2, 1, 0, 7, 6, 5, 4 }},
                    /*11111000*/{ CTBits5SameFace, &MarchingCubes::Bits5SameFace, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11111001*/{ CTBits6FaceDiag, &MarchingCubes::Bits6FaceDiag, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*11111010*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*11111011*/{ CTBits7, &MarchingCubes::Bits7, { 2, 0, 3, 1, 6, 4, 7, 5 }},
                    /*11111100*/{ CTBits6Edge, &MarchingCubes::Bits6Edge, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11111101*/{ CTBits7, &MarchingCubes::Bits7, { 1, 3, 0, 2, 5, 7, 4, 6 }},
                    /*11111110*/{ CTBits7, &MarchingCubes::Bits7, { 0, 1, 2, 3, 4, 5, 6, 7 }},
                    /*11111111*/{ CTBits0, &MarchingCubes::Bits0, { 0, 1, 2, 3, 4, 5, 6, 7 }}
            } };
            return configuration;
        }

        static std::array<std::string, CTNumTypes>& ConfigurationString()
        {
            static std::array<std::string, CTNumTypes> configurationString =
            {
                "Bits0",
                "Bits1",
                "Bits7",
                "Bits2Edge",
                "Bits6Edge",
                "Bits2FaceDiag",
                "Bits6FaceDiag",
                "Bits2BoxDiag",
                "Bits6BoxDiag",
                "Bits3SameFace",
                "Bits5SameFace",
                "Bits3EdgeFaceDiag",
                "Bits5EdgeFaceDiag",
                "Bits3FaceDiagFaceDiag",
                "Bits5FaceDiagFaceDiag",
                "Bits4SameFace",
                "Bits4FaceEdge",
                "Bits4FaceFaceDiagL",
                "Bits4FaceFaceDiagR",
                "Bits4FaceBoxDiag",
                "Bits4EdgeEdgePara",
                "Bits4EdgeEdgePerp"
            };
            return configurationString;
        }
    };
}

